--|
--| [ Quick Selector ]
--|
--| @author			Quity - T. Florin
--| @Description:	Quick Selector - Main Module
--|
--| Copyright (C) - T. Florin, All Rights Reserved.
--|
QuickSelector = {
   Hud = {},
   Vem = {},
   Actions = {},
   ConfigGui = {},
   I18n = {},
   Utility = {},
   Dialogs = {},
};
source(g_currentModDirectory .. "QuickSelectorDialogs.lua");
source(g_currentModDirectory .. "QuickSelectorUtility.lua");
source(g_currentModDirectory .. "QuickSelectorActions.lua");
source(g_currentModDirectory .. "QuickSelectorConfigGui.lua");
addModEventListener(QuickSelector);
QuickSelector.modName = g_currentModName;
QuickSelector.modDir = g_currentModDirectory;
QuickSelector.assetsDir = g_currentModDirectory .. "assets/";
function QuickSelector.getDefaultNamespaces()
   local QS = QuickSelector;
   return QS, QS.Hud, QS.Vem, QS.Actions, QS.ConfigGui, QS.I18n, QS.Utility, QS.Dialogs;
end
function QuickSelector.init(QS, Hud, Vem, Actions, ConfigGui, I18n, Utility, Dialogs)
   QS.mousePosX = 0.5;
   QS.mousePosY = 0.5;
   QS.position = {
      generalSpacing = 0.01;
      bottomStartVehiclesList = 0, 
      bottomStartQuickButtons = 0.11,
      buttonsInfoText = 0.02,
   };
   QS.Const = {
      defaultGameColor = {0.238, 0.462, 0, 1}, 
      gameColorDarker = {0.08, 0.150, 0, 1},
      defaultGameColorShade0_1 = {0.08, 0.150, 0, 0.1},
      defaultGameColorShade0_2 = {0.08, 0.150, 0, 0.2},
      inactiveIconColor = {0.034, 0.033, 0.034, 1},
      onHoverIconColor = {0.042, 0.068, 0, 1},
      activeClickColor = {0.02, 0.05, 0, 1},
      fullWhiteColor = {1, 1, 1, 1},
      fullWhiteColorShade0_5 = {1, 1, 1, 0.5},
      fullWhiteColorShade0_1 = {1, 1, 1, 0.1},
      fullBlackColor = {0, 0, 0, 1},
      transparentColor = {1, 1, 1, 0},
      closeBtnColor = {0.125, 0.012, 0.012, 1},
      quickButtonBgColor = {0.6, 0.6, 0.6, 1},
      MOUSE_LEFT_BUTTON = 1,
      MOUSE_MIDDLE_BUTTON = 2,
      MOUSE_RIGHT_BUTTON = 3,
      MOUSE_SCROLL_UP_BUTTON = 4,
      MOUSE_SCROLL_DOWN_BUTTON = 5,
      MOUSE_DEFAULT_ACTIVATION_BUTTON = 3,
      FIRST_LEVEL_SELECTOR_NR = 8,
      QS_INPUT_CONTEXT_NAME = "QuickSelectorHudInput",
      VEHICLE_BOX_PIXEL_SIZE = 85,
      InternalContexts = { MAIN = "MAIN", CONFIG_GUI = "CONFIG_GUI", INFO_DIALOG = "INFO_DIALOG", },
   };
   QS.Callbacks = {};
   Utility.init(QS, Utility);
   function I18n.init()
      local texts =  g_i18n.modEnvironments[QS.modName].texts; 
      function I18n:getName(name)
         return texts[name] and texts[name] or name;
      end
      function I18n:getInput(name)
         return texts["input_QUICKSELECTOR_"..name] and texts["input_QUICKSELECTOR_"..name] or name;
      end
      function I18n:getLabel(name)
         return texts["QUICKSELECTOR_LABEL_"..name] and texts["QUICKSELECTOR_LABEL_"..name] or name;
      end
      function I18n:getSetting(name)
         return texts["QS_SETTINGS_"..name] and texts["QS_SETTINGS_"..name] or name;
      end
   end
   function QS:toggleHud()
      if Hud.isVisible then
         QS:hideHud();
      else
         QS:showHud();
      end
   end
   function QS:showHud()
      if Hud.allowedContextOpening[g_inputBinding:getContextName()] then
         if Hud.shouldUpdateVehiclesList then
            Hud.shouldUpdateVehiclesList = false; 
            Hud:updateSelectableVehiclesList();
         else
            Hud:updateIsSelectedVehicle(); 
         end
         Actions.updateStates();
         Vem.exitReorderVehicleMode();
         Hud:updateVehicleInfoList();
         Hud:parseSelectedVehicleImplements();
         Hud:updateSelectorsIcon();
         Hud.isVisible = true;
         QS:lockInputContext(true);
         QS:resetInitialCursorPos();
      end
   end
   function QS:hideHud()
      if Hud.isVisible and Dialogs.activeDialog then
         Dialogs.activeDialog:close();
         return;
      end
      if Hud.allowedContextOpening[g_inputBinding:getContextName()] then
         QS:removeActionEvents();
         QS:lockInputContext(false);
         Hud.isVisible = false;
      end
   end
   function QS:lockInputContext(shouldLock)
      if Hud.isVisible and shouldLock then
         if not QS:isContextLocked() then
            g_inputBinding:setContext(QS.Const.QS_INPUT_CONTEXT_NAME, true, false);
            QS:registerActionEvents();
         end
         g_inputBinding:setShowMouseCursor(true);
      elseif Hud.isVisible and not shouldLock and QS:isContextLocked() then
         g_inputBinding:revertContext(true);
         g_inputBinding:setShowMouseCursor(false);
      end
   end
   function QS:revertLockedContext()
      if QS:isContextLocked() then
         g_inputBinding:revertContext(true);
      end
   end
   function QS:isContextLocked()
      return g_inputBinding:getContextName() == QS.Const.QS_INPUT_CONTEXT_NAME;
   end
   function QS:temporaryRevertLockedContext(callbackName, customDelay)
      QS:revertLockedContext();
      QS.Callbacks[callbackName] = function()
         QS:lockInputContext(true);
         Actions.updateStates();
      end
      addTimer(customDelay or 100, callbackName, QS.Callbacks); 
   end
   function QS:load()
      I18n.init();
      Dialogs.init(QS.getDefaultNamespaces());
      Hud.init(QS.getDefaultNamespaces());
      Vem.init(QS.getDefaultNamespaces());
      ConfigGui.init(QS.getDefaultNamespaces());
      Actions.init(QS.getDefaultNamespaces());
      Hud.createQuickButtons();
   end
   function QS:onGameFullLoaded()
      Hud:updateSelectableVehiclesList();
   end
   function QS:draw()
      if Hud.isVisible then
         Hud:renderOnScreen();
      end
   end
   function QS:loadMap()
      if Utility.addConsoleCommands then
         Utility:addConsoleCommands(); 
      end
      QS:load();
   end
   function onPlayerLoaded(player)
      if player.isOwner then
         QS:onGameFullLoaded();
         g_inputBinding:beginActionEventsModification(PlayerInputComponent.INPUT_CONTEXT_NAME); 
         QS:registerActionEventsHudToggle();
         g_inputBinding:endActionEventsModification();
      end
   end
   Player.load = Utils.appendedFunction(Player.load, onPlayerLoaded);
   function QS:mouseEvent(posX, posY, isDown, isUp, button)
      local wasHudVisible = Hud.isVisible;
      if ConfigGui.modConfig.rightMouseHoldDisplayHud and not ConfigGui.modConfig.doubleRightMouseClickDisplayHud then
         if QS.Const.MOUSE_DEFAULT_ACTIVATION_BUTTON == math.floor(button) and isDown and not Hud.isVisible then
            QS:showHud();
         elseif QS.Const.MOUSE_DEFAULT_ACTIVATION_BUTTON == math.floor(button) and isUp and Hud.isVisible then
            QS:hideHud();
         end
      end
      if ConfigGui.modConfig.doubleRightMouseClickDisplayHud and QS.Const.MOUSE_DEFAULT_ACTIVATION_BUTTON == math.floor(button) and isDown then
         if QS.rightMouseClickedOnce and not Hud.isVisible then
            QS.rightMouseClickedOnce = false;
            QS:showHud();
         elseif QS.rightMouseClickedOnce and Hud.isVisible then
            QS.rightMouseClickedOnce = false;
            QS:hideHud();
         else
            QS.rightMouseClickedOnce = true;
         end
         QS.resetRightMouseClickedOnceCB = function()
            QS.rightMouseClickedOnce = false;
         end
         addTimer(300, "resetRightMouseClickedOnceCB", QS);
      end
      if Hud.isVisible then
         if not wasHudVisible then
            wrapMousePosition(self.mousePosX, self.mousePosY); 
            return;
         end
         Hud:checkIfIsHovered(posX, posY, isDown, isUp, button);
         QS.mousePosX = posX;
         QS.mousePosY = posY;
      end
   end
   function QS.gamepadAnalogMoveEvent(_, eventName, value)
      local DEADZONE_LIMIT = ConfigGui.modConfig.gamepadDeadzoneLimit;
      local SPEED_MULTIPLIER = ConfigGui.modConfig.gamepadAnalogsSensitivity;
      local gamePadIsNotInDeadZone = (value > DEADZONE_LIMIT or value < -DEADZONE_LIMIT);
      if Hud.isVisible and ConfigGui.modConfig.enableGamepadControl and gamePadIsNotInDeadZone then
         local axisMultiplier = value / 10 * SPEED_MULTIPLIER;
         if (ConfigGui.modConfig.gamepadUseLeftAnalogToSteer and (eventName == "QUICKSELECTOR_GAMEPAD_ANALOG_AXIS_X" or eventName == "QUICKSELECTOR_GAMEPAD_ANALOG_AXIS_Y")) then
            return; 
         end
         if (eventName == "QUICKSELECTOR_GAMEPAD_ANALOG_AXIS_X" or eventName == "QUICKSELECTOR_GAMEPAD_RIGHT_ANALOG_AXIS_X") then
            QS.mousePosX = QS.mousePosX + (axisMultiplier / g_screenAspectRatio);
         else
            QS.mousePosY = QS.mousePosY - axisMultiplier;
         end
         QS.mousePosX = QS.mousePosX > 0.99 and 0.99 or QS.mousePosX;
         QS.mousePosX = QS.mousePosX < 0.01 and 0.01 or QS.mousePosX;
         QS.mousePosY = QS.mousePosY > 0.98 and 0.98 or QS.mousePosY;
         QS.mousePosY = QS.mousePosY < 0.02 and 0.02 or QS.mousePosY;
         wrapMousePosition(QS.mousePosX, QS.mousePosY); 
      end
   end
   function QS.gamepadClickEvent(_, _, keyStatus)
      local isDown = keyStatus == 1;
      local isUp = keyStatus == 0;
      Hud:checkIfIsHovered(QS.mousePosX, QS.mousePosY, isDown, isUp, QS.Const.MOUSE_LEFT_BUTTON);
   end
   function QS:resetInitialCursorPos()
      local resetToBottomPos = 0.35;
      local posXMax = 0.9;
      local posXMin = 0.1;
      local posYMax = 0.4;
      local posYMin = 0.01;
      if QS.mousePosX > posXMax or QS.mousePosX < posXMin or QS.mousePosY > posYMax or QS.mousePosY < posYMin then
         wrapMousePosition(0.5, resetToBottomPos);
      end
   end
   function Hud:setInternalContext(contextName)
      Hud.currentInternalContext = contextName;
   end
   function Hud:isItemInInternalContext(item)
      return not Hud.currentInternalContext or Hud.currentInternalContext == item.internalContext;
   end
   local SPATIAL_GRID_SIZE_X, SPATIAL_GRID_SIZE_Y = getNormalizedScreenValues(100, 100);  
   function Hud:getItemCellIndex(x, y)
      return math.floor(x / SPATIAL_GRID_SIZE_X), math.floor(y / SPATIAL_GRID_SIZE_Y);
   end
   function Hud:addItemToGrid(item)
      local leftCell, topCell = Hud:getItemCellIndex(item.posX, item.posY);
      local rightCell, bottomCell = Hud:getItemCellIndex(item.posX + item.width, item.posY + item.height);
      for x = leftCell, rightCell do
         for y = topCell, bottomCell do
            local cellKey = x .. "," .. y;
            Hud.itemsPosGrid[cellKey] = Hud.itemsPosGrid[cellKey] or {};
            table.insert(Hud.itemsPosGrid[cellKey], item);
         end
      end
   end
   function Hud:findHoveredItemInGrid(cursorX, cursorY)
      local cellX, cellY = Hud:getItemCellIndex(cursorX, cursorY);
      local cellKey = cellX .. "," .. cellY;
      local itemsInCell = Hud.itemsPosGrid[cellKey] or {};
      local itemFound;
      for _, item in ipairs(itemsInCell) do
         if item.isVisible and (not item.isChild or item.parent.isActive) and not item.isDisabled and Hud:isItemInInternalContext(item) and Hud:isPointInRect(cursorX, cursorY, item) then
            if not itemFound or item.zIndex > itemFound.zIndex then
               itemFound = item;
            end
         end
      end
      return itemFound;
   end
   function Hud:removeItemsFromGrid(items)
      for _, targetItem in pairs(items) do
         for cellKey, cellItems in pairs(Hud.itemsPosGrid) do
            for i = #cellItems, 1, -1 do
               if cellItems[i] == targetItem then
                  table.remove(cellItems, i)
                  break
               end
            end
            if #cellItems == 0 then
               Hud.itemsPosGrid[cellKey] = nil
            end
            if targetItem.childList then
               Hud:removeItemsFromGrid(targetItem.childList);
            end
         end
      end
   end
   function Hud:isPointInRect(cursorX, cursorY, item)
      return cursorX >= item.posX and cursorX <= item.posX + item.width and cursorY >= item.posY and cursorY <= item.posY + item.height;
   end
   function Hud:checkIfIsHovered(cursorX, cursorY, isDown, _, button)
      function _resetBtnUnHoverTimer(btn)
         if btn.unHoverTimer then
            removeTimer(btn.unHoverTimer);
            btn.unHoverTimer = false;
         end
      end
      function _resetLastActiveParentBtn(btn)
         if Hud.lastActiveBtn then
            _onDeactivateBtn(Hud.lastActiveBtn);
            _resetBtnUnHoverTimer(Hud.lastActiveBtn);
         end
         Hud.lastActiveBtn = btn;
      end
      function _onHoverBtn(btn)
         btn.isHovered = true;
         if btn.onHoverCallback then
            btn:onHoverCallback();
         end
      end
      function _onLeftHoverBtn(btn)
         btn.isHovered = false;
         if btn.onLeftHoverCallback then
            btn:onLeftHoverCallback();
         end
      end
      function _onDeactivateBtn(btn)
         btn.isActive = false;
         if btn.onDeactivateCallback then
            btn:onDeactivateCallback();
         end
      end
      function _onClick(btn)
         btn:onClickCallback();
      end
      local itemFound = Hud:findHoveredItemInGrid(cursorX, cursorY);
      if itemFound and itemFound.isChild and not itemFound.parent.isActive then
         itemFound = nil;
      end
      if itemFound then
         if not itemFound.isChild then
            _resetLastActiveParentBtn(itemFound); 
         else
            itemFound.parent.isActive = true;
            _resetBtnUnHoverTimer(itemFound.parent);
         end
         itemFound.isActive = true;
         _onHoverBtn(itemFound); 
         if isDown and button == QS.Const.MOUSE_LEFT_BUTTON then
            _onClick(itemFound);
         end
      end
      local leftHoverFound = (itemFound ~= Hud.currentHoveredItem and Hud.currentHoveredItem) and Hud.currentHoveredItem or false;
      if leftHoverFound and leftHoverFound.isHovered then
         if (not Hud.currentHoveredItem.hasChildSelector and not itemFound) or (itemFound and (itemFound ~= Hud.currentHoveredItem) and itemFound.parent ~= Hud.currentHoveredItem) then
            _onDeactivateBtn(Hud.currentHoveredItem); 
         end
         _onLeftHoverBtn(Hud.currentHoveredItem);
         local parentToKeepActive = leftHoverFound.isChild and leftHoverFound.parent or leftHoverFound;
         parentToKeepActive.leftHoverTimerCallback = function()
            _onDeactivateBtn(parentToKeepActive);
            parentToKeepActive.unHoverTimer = false;
         end
         _resetBtnUnHoverTimer(parentToKeepActive);
         parentToKeepActive.unHoverTimer = addTimer(600, "leftHoverTimerCallback", parentToKeepActive);
      end
      Hud.currentHoveredItem = itemFound and itemFound or Hud.currentHoveredItem;
   end
   QS.hudActionEvents = {};
   function QS:registerActionEventsHudToggle()
      _, QS.actionToggleHud = g_inputBinding:registerActionEvent(InputAction.QUICKSELECTOR_SHOW_HUD, QS, QS.toggleHud, false, true, false, true);
      g_inputBinding:setActionEventTextPriority(QS.actionToggleHud, GS_PRIO_VERY_HIGH);
      g_inputBinding:setActionEventTextVisibility(QS.actionToggleHud, true);
   end
   function QS:registerActionEvents()
      local ActEvs = QS.hudActionEvents;
      if Hud.isVisible then
         QS:registerActionEventsHudToggle(); 
         g_inputBinding:setActionEventText(QS.actionToggleHud, I18n:getInput("HIDE_HUD"));
         _, ActEvs.hideHud = g_inputBinding:registerActionEvent(InputAction.QUICKSELECTOR_HIDE_HUD, QS, QS.hideHud, false, true, false, true);
         if ConfigGui.modConfig.enableGamepadControl then
            _, ActEvs.selectElement = g_inputBinding:registerActionEvent(InputAction.QUICKSELECTOR_CLICK_HUD_ELEMENT, QS, QS.gamepadClickEvent, false, true, false, true); 
            g_inputBinding:setActionEventTextPriority(ActEvs.selectElement, GS_PRIO_VERY_HIGH);
            g_inputBinding:setActionEventTextVisibility(ActEvs.selectElement, true);
            g_inputBinding:setActionEventText(ActEvs.selectElement, I18n:getInput("CLICK_HUD_ELEMENT"));
            _, ActEvs.prevVehiclePage = g_inputBinding:registerActionEvent(InputAction.QUICKSELECTOR_GAMEPAD_PREV_VEHICLE_PAGE, QS, Hud.prevVehiclesListPage, false, true, false, true);
            _, ActEvs.nextVehiclePage = g_inputBinding:registerActionEvent(InputAction.QUICKSELECTOR_GAMEPAD_NEXT_VEHICLE_PAGE, QS, Hud.nextVehiclesListPage, false, true, false, true);
            local analogSticks = {InputAction.QUICKSELECTOR_GAMEPAD_ANALOG_AXIS_X, InputAction.QUICKSELECTOR_GAMEPAD_ANALOG_AXIS_Y, InputAction.QUICKSELECTOR_GAMEPAD_RIGHT_ANALOG_AXIS_X, InputAction.QUICKSELECTOR_GAMEPAD_RIGHT_ANALOG_AXIS_Y};
            for _, analogStickAxis in pairs(analogSticks) do
               _, ActEvs[analogStickAxis] = g_inputBinding:registerActionEvent(analogStickAxis, QS, QS.gamepadAnalogMoveEvent, false, false, true, true);
               g_inputBinding:setActionEventTextVisibility(ActEvs[analogStickAxis], false);
            end
         end
         local gameActionsList = {InputAction.MENU, InputAction.PAUSE, InputAction.TOGGLE_STORE};
         for _, value in pairs(gameActionsList) do
            _, ActEvs[value] = g_inputBinding:registerActionEvent(value, QS, QS.hideHud, false, true, false, true);
            g_inputBinding:setActionEventTextVisibility(ActEvs[value], false);
         end
         if QS.currentVehicleContext and QS.currentVehicleContext:getIsActiveForInput(true, true) then
            _, ActEvs.vehicleAccelerate = g_inputBinding:registerActionEvent(InputAction.AXIS_ACCELERATE_VEHICLE, QS.currentVehicleContext, Drivable.actionEventAccelerate, false, false, true, true, nil);
            _, ActEvs.vehicleBrake = g_inputBinding:registerActionEvent(InputAction.AXIS_BRAKE_VEHICLE, QS.currentVehicleContext, Drivable.actionEventBrake, false, false, true, true, nil);
            _, ActEvs.vehicleSteer = g_inputBinding:registerActionEvent(InputAction.AXIS_MOVE_SIDE_VEHICLE, QS.currentVehicleContext,
               function(self, actionName, inputValue, callbackState, isAnalog, isMouse, deviceCategory, binding)
                  if ConfigGui.modConfig.gamepadUseLeftAnalogToSteer or not isAnalog then 
                     Drivable.actionEventSteer(self, actionName, inputValue, callbackState, isAnalog, isMouse, deviceCategory, binding);
                  end
               end,
            false, false, true, true, nil);
         end
      end
   end
   function QS:removeActionEvents()
      g_inputBinding:setActionEventText(QS.actionToggleHud, I18n:getInput("SHOW_HUD"));
      for _, event in pairs(QS.hudActionEvents) do
         g_inputBinding:removeActionEvent(event);
      end
   end
end
function QuickSelector.Hud.init(QS, Hud, Vem, Actions, ConfigGui, I18n, Utility, Dialogs)
   Hud.itemsPosGrid = {}; 
   Hud.allowedContextOpening = {VEHICLE=true, PLAYER=true, QuickSelectorHudInput=true};
   Hud.shouldUpdateVehiclesList = false;
   function Hud:createBtn(index, isChild, isVisible, isDisabled, posX, posY, width, height, widthPercent, zIndex, name, customProps)
      local btn = {
         isHovered = false,
         isSelected = false,
         index = index,
         isChild = isChild,
         isVisible = isVisible,
         isDisabled = isDisabled,
         posX = posX,
         posY = posY,
         width = width,
         height = height,
         widthPercent = widthPercent,
         zIndex = zIndex,
         name = name
      };
      for key, value in pairs(customProps) do
         btn[key] = value;
      end
      btn.onClickCallback = function(self)
         if not self.isSelected then
            addTimer(150, "onLeftClickCallback", self);
            self.isClickEffectOn = true;
         end
         if customProps.onClickCallback then customProps.onClickCallback(self); end
      end;
      btn.onLeftClickCallback = function(self)
         self.isClickEffectOn = nil;
         if customProps.onLeftClickCallback then customProps.onLeftClickCallback(self); end
      end;
      btn.renderBaseOverlays = function(self, infoBoxText)
         local defaultBgColor = self.bgColor or QS.Const.fullWhiteColor;
         local currentColor = self.isClickEffectOn and QS.Const.activeClickColor or ((self.isActive or self.isSelected) and QS.Const.defaultGameColor or defaultBgColor);
         self.overlayBg:setColor(table.unpack(currentColor));
         self.overlayBg:render();
         if self.overlayIcon then
            self.overlayIcon:render();
         end
         if self.isHovered and infoBoxText then
            Hud:renderHoveredItemDescription(self, infoBoxText);
         end
      end
      Hud:addItemToGrid(btn);
      return btn;
   end
   local commonQBtnStyles = {iconColor=QS.Const.inactiveIconColor, iconColorActive=QS.Const.gameColorDarker, iconColorOnHover=QS.Const.onHoverIconColor };
   Hud.quickButtonsList = {};
   Hud.quickButtonsListNamesMap = {};
   function Hud.createQuickButtons()
      local quickButtonsConfig = {
         {name="START_ENGINE",				prerequisite="prerequisiteVehicle", 			callback="startMotor", 				icon="vehicle/startMotor",    style={iconColor=commonQBtnStyles.iconColor, iconColorActive={0.3, 0.05, 0, 1}, iconColorOnHover={0.3, 0.05, 0, 1}}},
         {name="ENTER_EXIT_VEHICLE", 		prerequisite="prerequisiteEnterable", 		   callback="enterExitVehicle", 		icon="vehicle/enterVehicle",  style=commonQBtnStyles},
         {name="OPEN_AI_SCREEN",		 		prerequisite="prerequisiteVehicle", 			callback="openAIScreen", 			icon="vehicle/openAIScreen",  style=commonQBtnStyles},
         {name="HIRE_HELPER", 				prerequisite="prerequisiteVehicle", 			callback="activateHelper", 		icon="vehicle/helper",        style=commonQBtnStyles},
         {name="CRUISE_CONTROL_ACTIVATE", 			   prerequisite="prerequisiteVehicle",	   callback="activateCruiseControl", 	icon="vehicle/cruiseControlActivation", style=commonQBtnStyles,
          childButtons={
             {name="CRUISE_CONTROL_MAX", 				   prerequisite="prerequisiteVehicle", 	callback="cruiseControlMaxSpeed", 	icon="vehicle/cruiseControlMax",        style=commonQBtnStyles},
             {name="CRUISE_CONTROL_INCREASE_SPEED",   prerequisite="prerequisiteVehicle", 	callback="increaseCruiseControl", 	icon="vehicle/cruiseControlIncrease",   style=commonQBtnStyles},
             {name="CRUISE_CONTROL_DECREASE_SPEED",   prerequisite="prerequisiteVehicle", 	callback="decreaseCruiseControl", 	icon="vehicle/cruiseControlDecrease",   style=commonQBtnStyles}
          }
         },
         {name="IMPLEMENT_ATTACH_TOGGLE",	prerequisite="prerequisiteVehicle", 			callback="implementAttachToggle", 	icon="vehicle/attach",           style=commonQBtnStyles},
         {name="IMPLEMENT_POWER_ON", 		prerequisite="prerequisiteVehicle", 		   callback="implementPowerOn", 		   icon="vehicle/implementPower",   style=commonQBtnStyles},
         {name="IMPLEMENT_FOLDING", 		prerequisite="prerequisiteVehicle", 		   callback="implementFolding", 		   icon="vehicle/folding",          style=commonQBtnStyles},
         {name="IMPLEMENT_LOWERING",		prerequisite="prerequisiteVehicle", 		   callback="implementLowering", 		icon="vehicle/lowering",         style=commonQBtnStyles},
         {name="IMPLEMENT_UNLOAD",			prerequisite="prerequisiteVehicle", 			callback="implementUnload", 			icon="vehicle/unload",           style=commonQBtnStyles,
          childButtons={
             {name="IMPLEMENT_UNLOAD_HERE",	prerequisite="prerequisiteVehicle", 			callback="implementUnloadHere",   	icon="vehicle/unloadHere",       style=commonQBtnStyles}
          }
         },
         {name="LIGHTS_TOGGLE", 				prerequisite="prerequisiteVehicle", 			callback="lightsToggle", 				icon="lights/lightsToggle",      style=commonQBtnStyles,
          childButtons={
             {name="LIGHTS_ALL", 			prerequisite="prerequisiteVehicle",		 	   callback="lightsAll", 					icon="lights/lightsAll",         style=commonQBtnStyles},
             {name="LIGHTS_BACK", 		   prerequisite="prerequisiteVehicle", 			callback="lightsBack", 				   icon="lights/lightsBack",        style=commonQBtnStyles},
             {name="LIGHTS_FRONT_TOP", 	prerequisite="prerequisiteVehicle", 			callback="lightsFrontTop", 			icon="lights/lightsFrontTop",    style=commonQBtnStyles},
             {name="LIGHTS_FRONT", 		   prerequisite="prerequisiteVehicle", 			callback="lightsFront", 				icon="lights/lightsFront",       style=commonQBtnStyles},
             {name="LIGHTS_HIGH_BEAM",    prerequisite="prerequisiteVehicle", 			callback="lightsHighBeam", 			icon="lights/lightsHighBeam",    style={iconColor=QS.Const.inactiveIconColor, iconColorActive={0, 0, 1, 0.8}, iconColorOnHover={0, 0, 1, 0.8}}},
             {name="LIGHTS_BEACON", 		prerequisite="prerequisiteVehicle", 			callback="lightsBeacon", 				icon="lights/lightsBeacon",      style=commonQBtnStyles},
             {name="LIGHTS_HAZARD", 		prerequisite="prerequisiteVehicle", 			callback="lightsHazard", 				icon="lights/lightsHazard",      style=commonQBtnStyles}
          }
         },
         {name="VEHICLES_LIST_REORDER",	callback="reoderVehicleListMode",	      icon="general/reorderVehicleList",  style={iconColor=commonQBtnStyles.iconColor, iconColorActive={0.5, 0, 0, 1}, iconColorOnHover={0.5, 0, 0, 1}}},
         {name="OPEN_MENU",					callback="openMenu",								icon="general/openMenu",            style=commonQBtnStyles},
         {name="OPEN_SHOP",					callback="openShop",								icon="general/openShop",            style=commonQBtnStyles},
         {name="SAVE_GAME",					callback="saveGame",								icon="general/savegame",            style=commonQBtnStyles},
         {name="QUIT_GAME",					callback="quitGame",								icon="general/shutdown",            style=commonQBtnStyles},
         {name="SETTINGS",						callback="showConfig",						   icon="general/hudConfig",           style=commonQBtnStyles},
         {name="CLOSE_HUD",					callback="closeHud", 							icon="general/closeHud",            style={iconColor=QS.Const.closeBtnColor, iconColorOnHover=QS.Const.closeBtnColor}},
      };
      Hud:removeItemsFromGrid(Hud.quickButtonsList); 
      Hud.quickButtonsList = {};
      local HUD_SIZE_MULTIPLIER = ConfigGui.modConfig.hudSizeMultiplier;
      local widthMultiplied = 38 * HUD_SIZE_MULTIPLIER;
      local width, height = getNormalizedScreenValues(widthMultiplied, widthMultiplied * 1.06); 
      local widthPercent = width * 0.01;
      local heightPercent = height * 0.01;
      local SELECTOR_SPACING_X, SELECTOR_SPACING_Y = getNormalizedScreenValues(4 * HUD_SIZE_MULTIPLIER, 2.2 * HUD_SIZE_MULTIPLIER);
      local widthAndSpace = width + SELECTOR_SPACING_X;
      local startPosX = 0.5 - (widthAndSpace * #quickButtonsConfig / 2); 
      local startPosY = QS.position.bottomStartQuickButtons * HUD_SIZE_MULTIPLIER;
      QS.position.bottomStartVehiclesList = startPosY + height + QS.position.generalSpacing;
      function _parseQuickButtonChilds(parentBtn, parentBtnConfig)
         local childButtons = {};
         local childStartPosX = parentBtn.posX - ((#parentBtnConfig.childButtons - 1) * widthAndSpace / 2);
         local childStartPosY = parentBtn.posY - parentBtn.height - SELECTOR_SPACING_Y;
         parentBtn.hasChildSelector=false;
         for index, item in ipairs(parentBtnConfig.childButtons) do
            local childBtnInstance = _createQuickButton(item, index, childStartPosX, childStartPosY);
            childBtnInstance.isChild = true;
            childBtnInstance.parent = parentBtn;
            childButtons[index] = childBtnInstance;
            childStartPosX = childStartPosX + widthAndSpace;
            parentBtn.hasChildSelector=true;
         end
         return childButtons;
      end
      function _updateButtonColors(btn)
         btn:applyButtonState();
         btn.bgColor = btn.isClickEffectOn and QS.Const.activeClickColor or ((btn.isActive or btn.isSelected) and QS.Const.defaultGameColor or QS.Const.quickButtonBgColor);
         btn.barColor = btn.isClickEffectOn and QS.Const.activeClickColor or ((btn.isActive or btn.isSelected) and QS.Const.defaultGameColor or QS.Const.fullWhiteColor);
      end
      function _createQuickButton(btnItemConfig, index, posX, posY)
         posY = posY - SELECTOR_SPACING_Y;
         local iconNamePath = QS.assetsDir .. btnItemConfig.icon;
         local iconWidth = width - (widthPercent * 30);
         local iconHeight = iconWidth * g_screenAspectRatio;
         local iconPosX = posX + (widthPercent * 15);
         local iconPosY = posY + (heightPercent * 17);
         local _, barNormalizedHeight = getNormalizedScreenValues(0, 3 * HUD_SIZE_MULTIPLIER);
         local barPosX = posX;
         local barPosY = posY - barNormalizedHeight / 2;
         local barWidth = width;
         local barHeight = barNormalizedHeight;
         local quickBtnInstance = Hud:createBtn(index, false, true, false, posX, posY, width, height, widthPercent, 1, btnItemConfig.name, {
            bgColor = QS.Const.quickButtonBgColor,
            barColor = QS.Const.fullWhiteColor,
            applyButtonState = function(self)
               local activeColor = (self.isActive or self.isSelected) and btnItemConfig.style.iconColorOnHover or btnItemConfig.style.iconColorActive;
               self.overlayIcon:setColor(Utility.unpackColor(Actions.states[btnItemConfig.callback] and activeColor or btnItemConfig.style.iconColor));
            end,
            onClickCallback = function(self)
               if btnItemConfig.prerequisite then
                  Actions[btnItemConfig.prerequisite](self, btnItemConfig.callback); 
               else
                  Actions.general[btnItemConfig.callback](self, btnItemConfig.callback); 
               end
               _updateButtonColors(self);
            end,
            onLeftClickCallback = function(self)
               _updateButtonColors(self);
            end,
            onHoverCallback = function(self) _updateButtonColors(self); end,
            onLeftHoverCallback = function(self) _updateButtonColors(self); end,
            onDeactivateCallback = function(self) _updateButtonColors(self); end,
            renderQuickButton = function(self, infoBoxText)
               drawFilledRectRound(self.posX, self.posY, self.width, self.height, 0.08, Utility.unpackColor(self.bgColor, 0.4));  
               drawFilledRectRound(barPosX, barPosY, barWidth, barHeight, 0.02, Utility.unpackColor(self.barColor, 0.8));         
               self.overlayIcon:render();
               if self.isHovered and infoBoxText then
                  Hud:renderHoveredItemDescription(self, infoBoxText);
               end
            end,
            overlayIcon = Utility.newColoredOverlay(iconNamePath .. ".dds", iconPosX, iconPosY, iconWidth, iconHeight, btnItemConfig.style.iconColor);
         });
         if btnItemConfig.childButtons then
            quickBtnInstance.childList = _parseQuickButtonChilds(quickBtnInstance, btnItemConfig);
         end
         return quickBtnInstance;
      end
      for index, item in ipairs(quickButtonsConfig) do
         Hud.quickButtonsList[index] = _createQuickButton(item, index, startPosX, startPosY);
         Hud.quickButtonsListNamesMap[item.name] = Hud.quickButtonsList[index];
         startPosX = startPosX + widthAndSpace;
      end
   end
   Hud.vehicleInfoBoxesList = {};
   function Hud:updateVehicleInfoList(selectedVehicle)
      local currentVehicle = selectedVehicle and selectedVehicle or g_localPlayer:getCurrentVehicle();
      local calculatedTotalWidth = 0;
      local widthMultiplied = 30 * ConfigGui.modConfig.hudSizeMultiplier;
      local BOX_TEXT_SIZE = getCorrectTextSize(0.012) * ConfigGui.modConfig.hudSizeMultiplier;
      local BOX_PADDING = getNormalizedScreenValues(widthMultiplied, 0);
      local BOX_SPACING = getNormalizedScreenValues(widthMultiplied / 10, 0);
      Hud.vehicleInfoBoxesList = {};
      if not ConfigGui.modConfig.vehicleDisplayInfoLine or not currentVehicle or not Vem:isAllowedVehicleType(currentVehicle) then
         return;
      end
      function _getFormattedAge(config, vehicle)
         local monthOrYear = I18n:getLabel(vehicle.age < 12 and "MONTH" or "YEAR")
         return string.format(I18n:getLabel(config.label), vehicle.age < 12 and vehicle.age or math.floor((vehicle.age + 6) / 12), monthOrYear);
      end
      function _getMotorHP(vehicle)
         local motor = vehicle:getMotor();
         return (motor and motor.peakMotorPower or 0) * 1.35962; 
      end
      local vehicleInfoConfig = {
         { label="VI_FULL_NAME",  icon="general/openShop",        getValue=function(config, vehicle) return string.format(I18n:getLabel(config.label), vehicle:getFullName()); end },
         { label="VI_AGE",        icon="general/age",             getValue=function(config, vehicle) return _getFormattedAge(config, vehicle) end },
         { label="VI_SELL_VALUE", icon="general/sell",            getValue=function(config, vehicle) return string.format(I18n:getLabel(config.label), g_i18n:formatMoney(vehicle:getSellPrice(), 0)); end },
         { label="VI_ENGINE_HP",  icon="general/enginePower",     getValue=function(config, vehicle) return string.format(I18n:getLabel(config.label), _getMotorHP(vehicle)) end  },
         { label="VI_DAMAGE",     icon="lights/lightsHazard",     getValue=function(config, vehicle) return string.format(I18n:getLabel(config.label), vehicle:getVehicleDamage() * 100); end },
         { label="VI_DIRT",       icon="general/dirt",            getValue=function(config, vehicle) return string.format(I18n:getLabel(config.label), vehicle:getDirtAmount() * 100); end },
      };
      function _updateInfoBoxValueAndCalculateWidth(boxItemConfig)
         boxItemConfig.updatedValue = boxItemConfig.getValue(boxItemConfig, currentVehicle);
         local textWidth = getTextWidth(BOX_TEXT_SIZE, boxItemConfig.updatedValue);
         local textHeight = getTextHeight(BOX_TEXT_SIZE, boxItemConfig.updatedValue);
         boxItemConfig.width = textWidth + BOX_PADDING * 1.2;
         boxItemConfig.height = textHeight + BOX_PADDING;
         calculatedTotalWidth = calculatedTotalWidth + boxItemConfig.width + BOX_SPACING;
      end
      function _createVehicleInfoBox(index, boxItemConfig, nextPosX, width, height)
         local heightPercent = height * 0.01;
         local iconHeight = height - (heightPercent * 22);
         local iconWidth = iconHeight / g_screenAspectRatio;
         local startPosY = 0.005 + self.vehicleListTopEndPosY + BOX_SPACING;
         local iconPosX = nextPosX + (heightPercent * 10);
         local iconPosY = startPosY + (heightPercent * 11);
         Hud.vehicleInfoBoxTopEndPosY = startPosY + height;
         local box = {
            index = index,
            posX = nextPosX,
            posY = startPosY,
            width = width,
            height = height,
            label = boxItemConfig.updatedValue,
            textSize = BOX_TEXT_SIZE,
            textPaddingLeft = (iconWidth / 2);
            textColor = QS.Const.fullWhiteColor,
            overlayIcon = Utility.newColoredOverlay(QS.assetsDir .. boxItemConfig.icon .. ".dds", iconPosX, iconPosY, iconWidth, iconHeight, QS.Const.defaultGameColor),
            renderInfoBox = function(self)
               drawFilledRectRound(self.posX, self.posY - heightPercent * 3, self.width, self.height, 0.1, Utility.unpackColor(QS.Const.fullBlackColor, 0.8));    
               drawFilledRectRound(self.posX, self.posY - heightPercent * 6, self.width, heightPercent * 12, 0.02, Utility.unpackColor({0.01, 0.01, 0.01, 1}));   
               self.overlayIcon:render();
               Hud:renderBoxLabel(self);
            end
         }
         return box;
      end
      for _, config in ipairs(vehicleInfoConfig) do
         _updateInfoBoxValueAndCalculateWidth(config);
      end
      local nextPosX = 0.5 - (calculatedTotalWidth / 2); 
      for index, config in ipairs(vehicleInfoConfig) do
         Hud.vehicleInfoBoxesList[index] = _createVehicleInfoBox(index, config, nextPosX, config.width, config.height);
         nextPosX = nextPosX + config.width + BOX_SPACING;
      end
   end
   Hud.vehiclesList = {};
   function Hud:changeVehiclesListPage(isForward, currentPageToKeep)
      local lastPageNumber = math.floor((Hud.vehiclesCounter - 1) / ConfigGui.modConfig.vehicleListMaxPerPage) + 1;
      if isForward and Hud.vehiclesListCurrentPage < lastPageNumber then
         Hud.vehiclesListCurrentPage = Hud.vehiclesListCurrentPage + 1;
      elseif not isForward and Hud.vehiclesListCurrentPage > 1 then
         Hud.vehiclesListCurrentPage = Hud.vehiclesListCurrentPage - 1;
      end
      Hud.vehiclesListCurrentPage = currentPageToKeep and currentPageToKeep or Hud.vehiclesListCurrentPage;
      Hud.leftPaginationArrowBtn.isDisabled = Hud.vehiclesListCurrentPage == 1;
      Hud.rightPaginationArrowBtn.isDisabled = Hud.vehiclesListCurrentPage == lastPageNumber;
   end
   function Hud:nextVehiclesListPage()
      if not Hud.rightPaginationArrowBtn.isDisabled then
         Hud:changeVehiclesListPage(true);
      end
   end
   function Hud:prevVehiclesListPage()
      if not Hud.leftPaginationArrowBtn.isDisabled then
         Hud:changeVehiclesListPage();
      end
   end
   function Hud:getVehiclesInCurrentPage(callback)
      local vehicleListEndIndex = Hud.vehiclesListCurrentPage * ConfigGui.modConfig.vehicleListMaxPerPage;
      local vehicleListStartIndex = vehicleListEndIndex - ConfigGui.modConfig.vehicleListMaxPerPage;
      for key, vehicle in pairs(Hud.vehiclesList) do
         vehicle.isVisible = false; 
         if key > vehicleListStartIndex and key <= vehicleListEndIndex then
            vehicle.isVisible = true; 
            callback(key, vehicle);
         end
      end
   end
   function Hud:getItemImagePath(basePath, imagePath)
      if basePath and imagePath and string.find(imagePath, "$data") then
         basePath = ""; 
      end
      return basePath .. string.gsub(Utils.getNoNil(imagePath, ""), "%$", ""); 
   end
   function Hud:getIsSelectedVehicle(vehicle)
      local currentVehicle = g_localPlayer and g_localPlayer:getCurrentVehicle() or nil;
      local isSelected = (vehicle.isImplement and vehicle.instance:getIsSelected()) or (currentVehicle and currentVehicle.id == vehicle.instance.id);
      if isSelected then
         if Hud.currentSelectedVehicle then
            Hud.currentSelectedVehicle.isSelected = false;
         end
         Hud.currentSelectedVehicle = vehicle;
      end
      return isSelected;
   end
   function Hud:updateIsSelectedVehicle()
      for _, vehicle in pairs(Hud.vehiclesList) do
         vehicle.isSelected = Hud:getIsSelectedVehicle(vehicle);
      end
   end
   function Hud:updateSelectableVehiclesList(keepCurrentPage)
      local SELECTOR_SPACING = getNormalizedScreenValues(-0.0004 * g_referenceScreenWidth * ConfigGui.modConfig.hudSizeMultiplier, 0);
      local widthMultiplied = QS.Const.VEHICLE_BOX_PIXEL_SIZE * ConfigGui.modConfig.hudSizeMultiplier;
      local width, height = getNormalizedScreenValues(widthMultiplied, widthMultiplied * 1.4); 
      local widthPercent = width * 0.01;
      local heightPercent = height * 0.01;
      local widthAndSpacing = width + SELECTOR_SPACING;
      local btnWidth = width / 2;
      local btnHeight = height / 2;
      local filteredVehicleListLen = 0;
      for _, vehicle in pairs(g_currentMission.vehicleSystem.enterables) do
         if Vem:isAllowedVehicleType(vehicle) then
            filteredVehicleListLen = filteredVehicleListLen + 1;
         end
      end
      local vehicleCountInPage = filteredVehicleListLen < ConfigGui.modConfig.vehicleListMaxPerPage and filteredVehicleListLen or ConfigGui.modConfig.vehicleListMaxPerPage;
      local startPosX = 0.5 - (widthAndSpacing * vehicleCountInPage / 2);
      local startPosY = QS.position.bottomStartVehiclesList;
      local leftPagBtnPosX = startPosX - btnWidth;
      local rightPagBtnPosX = startPosX + (widthAndSpacing * vehicleCountInPage);
      Hud:removeItemsFromGrid(Hud.vehiclesList); 
      Hud.vehiclesList = {};
      Hud.vehiclesCounter = 0;
      Hud.vehiclesListCurrentPage = keepCurrentPage and Hud.vehiclesListCurrentPage or 1;
      Hud.vehicleListTopEndPosY = startPosY + height;
      Hud.vehicleInfoBoxTopEndPosY = Hud.vehicleListTopEndPosY + SELECTOR_SPACING; 
      Hud:removeItemsFromGrid({Hud.leftPaginationArrowBtn, Hud.rightPaginationArrowBtn});
      function _createPaginationBtn(btnId, isForward, posX, posY, width, height)
         Hud[btnId] = Hud:createBtn(0, false, true, true, posX, posY, width, height, nil, 1, "", {
            onClickCallback = function()
               Hud:changeVehiclesListPage(isForward);
            end,
            overlayBg = Overlay.new(QS.assetsDir .. "arrowLeft.dds", posX, posY, width, height),
         });
      end
      _createPaginationBtn("leftPaginationArrowBtn", false, leftPagBtnPosX, startPosY + (btnHeight / 2), btnWidth, btnHeight);
      _createPaginationBtn("rightPaginationArrowBtn", true, rightPagBtnPosX, startPosY + (btnHeight / 2), btnWidth, btnHeight);
      Hud.rightPaginationArrowBtn.overlayBg:setRotation(math.rad(180), btnWidth / 2, btnHeight / 2);
      function _createVehicleSelector(vehicle, index, counter, posX, posY, width, height)
         local vehicleConfigXml = loadXMLFile("vehicleConfig", vehicle.configFileName);
         local vehicleImage = Hud:getItemImagePath(vehicle.baseDirectory, getXMLString(vehicleConfigXml, "vehicle.storeData.image"));
         local iconWidth = width - (widthPercent * 4);
         local iconHeight = iconWidth * g_screenAspectRatio;
         local iconPosX = posX - (widthPercent * 2);
         local iconPosY = posY + (heightPercent * 32);
         delete(vehicleConfigXml);
         local vehicleInstance = Hud:createBtn(index, false, false, false, posX, posY, width, height, widthPercent, 1, vehicle:getName(), {
            isVehicle = true,
            counter = counter,
            image = vehicleImage,
            instance = vehicle,
            heightPercent = heightPercent,
            onClickCallback = function(self)
               if Actions.states.reoderVehicleListMode then
                  Vem:onVehiclePositionChangeClick(self);
                  return;
               end
               if not self.isSelected then
                  QS:revertLockedContext();
                  g_localPlayer:requestToEnterVehicle(self.instance);
                  self.isSelected = Hud:getIsSelectedVehicle(self);
               end
            end,
            overlayBg = Overlay.new(QS.assetsDir .. "vehicleListBg.dds", posX, posY, width, height);
            overlayIcon = Overlay.new(vehicleImage, iconPosX, iconPosY, iconWidth, iconHeight);
         });
         vehicleInstance.isSelected = Hud:getIsSelectedVehicle(vehicleInstance);
         return vehicleInstance;
      end
      for index, vehicle in pairs(g_currentMission.vehicleSystem.enterables) do
         if Vem:isAllowedVehicleType(vehicle) then
            Hud.vehiclesCounter = Hud.vehiclesCounter + 1;
            local resetCounterInPage = (math.fmod(Hud.vehiclesCounter - 1, ConfigGui.modConfig.vehicleListMaxPerPage)); 
            local nextPosX = startPosX + (widthAndSpacing * resetCounterInPage);
            Hud.vehiclesList[Hud.vehiclesCounter] = _createVehicleSelector(vehicle, index, Hud.vehiclesCounter, nextPosX, startPosY, width, height); 
         end
      end
      Hud:changeVehiclesListPage(true, Hud.vehiclesListCurrentPage);
   end
   function Hud:getImplementsList(callback)
      for key, implement in pairs(Hud.implementsList) do
         callback(key, implement);
      end
   end
   function Hud:getIsSelectedImplement(implement)
      local isSelected = implement.instance:getIsSelected();
      if isSelected then
         if Hud.currentSelectedImplement then
            Hud.currentSelectedImplement.isSelected = false;
         end
         Hud.currentSelectedImplement = implement;
      end
      return isSelected;
   end
   Hud.implementsList = {};
   function Hud:parseSelectedVehicleImplements()
      local _tempImplementsList = {};
      local currentVehicle = g_localPlayer:getCurrentVehicle();
      local attachedImplementsList = Vem:isAllowedVehicleType(currentVehicle) and currentVehicle:getAttachedImplements() or nil;
      Hud:removeItemsFromGrid(Hud.implementsList); 
      Hud.implementsList = {};
      if attachedImplementsList and next(attachedImplementsList) then
         local IMPLEMENTS_COUNTER = 1;
         local SELECTOR_SPACING = getNormalizedScreenValues(-0.0004 * g_referenceScreenWidth * ConfigGui.modConfig.hudSizeMultiplier, 0);
         local widthMultiplied = 77 * ConfigGui.modConfig.hudSizeMultiplier;
         local width, height = getNormalizedScreenValues(widthMultiplied, widthMultiplied * 1.2); 
         local widthPercent = width * 0.01;
         local heightPercent = height * 0.01;
         local startPosY = 0.005 + Hud.vehicleInfoBoxTopEndPosY + SELECTOR_SPACING;
         function _changeSelectedImplement(implement)
            QS:revertLockedContext();
            QS:temporaryRevertLockedContext("onImplementEnterTimer"); 
            implement.parentVehicleInstance:setSelectedVehicle(implement.instance, nil, true);
            implement.isSelected = Hud:getIsSelectedImplement(implement);
         end
         function _createImplementInstance(implementRoot, index, posX, posY)
            local implement = implementRoot.object;
            local iconWidth = width - (widthPercent * 22);
            local iconHeight = iconWidth * g_screenAspectRatio;
            local iconPosX = posX + (widthPercent * 6);
            local iconPosY = posY + (heightPercent * 36);
            local implementConfigXml = loadXMLFile("vehicleConfig", implement.configFileName);
            local implementImage = Hud:getItemImagePath(implement.baseDirectory, getXMLString(implementConfigXml, "vehicle.storeData.image"));
            delete(implementConfigXml);
            local implementInstance = Hud:createBtn(index, false, true, false, posX, posY, width, height, widthPercent, 1, implement:getName(), {
               isImplement = true,
               image = implementImage,
               instance = implement,
               implementRoot = implementRoot,
               parentVehicleInstance = currentVehicle,
               heightPercent = heightPercent,
               onClickCallback = function(self)
                  _changeSelectedImplement(self);
               end,
               overlayBg = Overlay.new(QS.assetsDir .. "vehicleListBg.dds", posX, posY, width, height);
               overlayIcon = Overlay.new(implementImage, iconPosX, iconPosY, iconWidth, iconHeight);
            });
            implementInstance.isSelected = Hud:getIsSelectedImplement(implementInstance);
            local barWidth = g_pixelSizeX * 2;
            local barHeight = height - (heightPercent * 20);
            local barPosX = posX + width - (widthPercent * 12);
            local barPosY = posY + (heightPercent * 10);
            local barBgOffset = barWidth;
            local implementDamage = implement.getDamageAmount and implement:getDamageAmount() or 1; 
            local barDamageValue = barHeight - (implementDamage * barHeight);
            local barDamageColor = implementDamage > 0.8 and SpeedSliderDisplay.COLOR.NEGATIVE_BAR or SpeedSliderDisplay.COLOR.POSITIVE_BAR;
            implementInstance.overlayInstanceDamageBg = Overlay.new(g_baseUIFilename, barPosX, barPosY - barBgOffset, barWidth, barHeight + (barBgOffset * 2));
            implementInstance.overlayInstanceDamageBg:setUVs(g_colorBgUVs);
            implementInstance.overlayInstanceDamageBg:setColor(250, 250, 250, 0.5);
            implementInstance.overlayInstanceDamage = Overlay.new(g_baseUIFilename, barPosX, barPosY, barWidth, barDamageValue);
            implementInstance.overlayInstanceDamage:setUVs(g_colorBgUVs);
            implementInstance.overlayInstanceDamage:setColor(barDamageColor[1], barDamageColor[2], barDamageColor[3], 0.6);
            return implementInstance;
         end
         function _iterateAttachedImplementsList(attachedImplementsList, isRootItem, position)
            for _, implementRoot in pairs(attachedImplementsList) do
               if (implementRoot.IS_INSERTED_VEHICLE or not implementRoot.object.spec_attachable) or not implementRoot.object.spec_attachable.detachingInProgress then
                  implementRoot.IS_FRONT_IMPLEMENT = false;
                  implementRoot.IS_BACK_IMPLEMENT = false;
                  local jointDesc = currentVehicle.schemaOverlay.attacherJoints[implementRoot.jointDescIndex];
                  local isFrontImplement = jointDesc and jointDesc.invertX or false;
                  if not isRootItem then
                     isFrontImplement = position == "IS_FRONT_IMPLEMENT" and true or false;
                  end
                  if position == "IS_BACK_IMPLEMENT" and not isFrontImplement and implementRoot.object:getAttachedImplements() then
                     _iterateAttachedImplementsList(implementRoot.object:getAttachedImplements(), false, position); 
                  end
                  if (position == "IS_INSERTED_VEHICLE") or (position == "IS_BACK_IMPLEMENT" and not isFrontImplement) or (position == "IS_FRONT_IMPLEMENT" and isFrontImplement) then
                     _tempImplementsList[IMPLEMENTS_COUNTER] = implementRoot;
                     _tempImplementsList[IMPLEMENTS_COUNTER][position] = true;
                     IMPLEMENTS_COUNTER = IMPLEMENTS_COUNTER + 1;
                  end
                  if position == "IS_FRONT_IMPLEMENT" and isFrontImplement and implementRoot.object:getAttachedImplements() then
                     _iterateAttachedImplementsList(implementRoot.object:getAttachedImplements(), false, position); 
                  end
               end
            end
         end
         function _createImplementSelectorsInstance()
            local startPosX = 0.5 - (((width + SELECTOR_SPACING) * (IMPLEMENTS_COUNTER + 1)) / 2); 
            for index, implementRoot in pairs(_tempImplementsList) do
               local nextPosX = startPosX + ((width + SELECTOR_SPACING) * index);
               Hud.implementsList[index] = _createImplementInstance(implementRoot, index, nextPosX, startPosY);
            end
            _tempImplementsList = nil;
         end
         _iterateAttachedImplementsList(attachedImplementsList, true,"IS_BACK_IMPLEMENT");
         _iterateAttachedImplementsList({{object=currentVehicle, IS_INSERTED_VEHICLE=true}}, true,"IS_INSERTED_VEHICLE"); 
         _iterateAttachedImplementsList(attachedImplementsList, true, "IS_FRONT_IMPLEMENT");
         _createImplementSelectorsInstance(); 
      end
   end
   function _resetTextDefaultValues()
      setTextAlignment(RenderText.ALIGN_LEFT);
      setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_BASELINE);
      setTextBold(false);
   end
   function _renderVehicleName(vehicle, paddingPercent)
      local heightPercentMultiplier = vehicle.isImplement and 22 or 19;
      local textSize = getCorrectTextSize(0.010) * ConfigGui.modConfig.hudSizeMultiplier;
      local textPosX = vehicle.posX + (vehicle.widthPercent * 50);
      local textPosY = vehicle.posY + (vehicle.heightPercent * heightPercentMultiplier);
      setTextColor(table.unpack(QS.Const.fullWhiteColor));
      setTextWrapWidth(vehicle.width - (vehicle.widthPercent * paddingPercent));
      setTextLineBounds(0, 2);
      setTextAlignment(RenderText.ALIGN_CENTER);
      setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_MIDDLE);
      setTextBold(true);
      local height, numLines = getTextHeight(textSize, vehicle.name);
      if numLines == 1 then
         textPosY = textPosY - (height / 4);
      end
      renderText(textPosX, textPosY, textSize, vehicle.name);
      setTextWrapWidth(0);
      setTextLineBounds(0, 0)
   end
   function Hud:renderBoxLabel(box)
      local paddingLeft = box.textPaddingLeft and box.textPaddingLeft or 0; 
      local textPosX = box.posX + (box.width / 2) + paddingLeft;
      local textPosY = box.posY + (box.height / 2);
      setTextAlignment(RenderText.ALIGN_CENTER);
      setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_MIDDLE);
      setTextBold(true);
      setTextColor(table.unpack(box.textColor));
      renderText(textPosX, textPosY, box.textSize, I18n:getLabel(box.label));
   end
   function Hud:renderHoveredItemDescription(button, text)
      local textSize = getCorrectTextSize(0.011) * ConfigGui.modConfig.hudSizeMultiplier;
      local textPosX = 0.5;
      local textPosY = QS.position.buttonsInfoText;
      local textPadding = 0.01;
      local textWidth = getTextWidth(textSize, text);
      local textHeight, _ = getTextHeight(textSize, text);
      local textBgWidth = textWidth + textPadding;
      local textBgHeight = textHeight + textPadding;
      local textBgPosX = textPosX - (textBgWidth / 2);
      local textBgPosY = textPosY - (textBgHeight / 2) + 0.001;
      drawFilledRectRound(textBgPosX, textBgPosY, textBgWidth, textBgHeight, 0.1, 0.01, 0.01, 0.01, 0.8); 
      Hud:renderBoxLabel({posX=textBgPosX, posY=textBgPosY, width=textBgWidth, height=textBgHeight, label=text, textSize=textSize, textColor=QS.Const.fullWhiteColor});
   end
   function Hud:renderOnScreen()
      if Dialogs.activeDialog then
         Dialogs:render();
         _resetTextDefaultValues();
         return; 
      end
      function _renderQuickButtonsList(list)
         for _, quickBtn in ipairs(list) do
            quickBtn:renderQuickButton(I18n:getLabel(quickBtn.name));
            if quickBtn.hasChildSelector and (quickBtn.isActive or (quickBtn.isChild and quickBtn.parent.isActive)) then
               _renderQuickButtonsList(quickBtn.childList); 
            end
         end
      end
      _renderQuickButtonsList(Hud.quickButtonsList);
      Hud:getImplementsList(function(_, implement)
         implement:renderBaseOverlays(implement.instance:getFullName());
         implement.overlayInstanceDamageBg:render();
         implement.overlayInstanceDamage:render();
         _renderVehicleName(implement, 30);
      end);
      function _renderVehicleInfoBoxes(list)
         for _, box in ipairs(list) do
            box:renderInfoBox();
         end
      end
      _renderVehicleInfoBoxes(Hud.vehicleInfoBoxesList);
      Hud:getVehiclesInCurrentPage(function(_, vehicle)
         vehicle:renderBaseOverlays(vehicle.instance:getFullName());
         _renderVehicleName(vehicle, 10);
         Vem:renderVehiclePositionChangeInfo(vehicle); 
      end);
      if not Hud.leftPaginationArrowBtn.isDisabled then
         Hud.leftPaginationArrowBtn.overlayBg:render();
      end
      if not Hud.rightPaginationArrowBtn.isDisabled then
         Hud.rightPaginationArrowBtn.overlayBg:render();
      end
      _resetTextDefaultValues();
   end
   function Hud:updateSelectorsIcon()
      for _, quickBtn in pairs(Hud.quickButtonsList) do
         quickBtn:applyButtonState();
         if quickBtn.hasChildSelector then
            for _, childQuickBtn in pairs(quickBtn.childList) do
               childQuickBtn:applyButtonState();
            end
         end
      end
   end
end
function QuickSelector.Vem.init(QS, Hud, Vem, Actions, ConfigGui, I18n, Utility)
   function Vem:isAllowedVehicleType(vehicle)
      return vehicle and vehicle.getIsTabbable and vehicle:getIsTabbable() and g_currentMission.accessHandler:canPlayerAccess(vehicle) and (vehicle.typeName and vehicle.typeName ~= "horse" and vehicle.typeName ~= "locomotive");
   end
   function Vem:toggleReoderVehiclesList(btn)
      if Actions.states.reoderVehicleListMode then
         btn.name = I18n:getLabel("VEHICLES_LIST_REORDER_EXIT");
      else
         Vem:exitReorderVehicleMode();
      end
   end
   function Vem:exitReorderVehicleMode()
      Actions.states.reoderVehicleListMode = false;
      Hud.quickButtonsListNamesMap.VEHICLES_LIST_REORDER.name = I18n:getLabel("VEHICLES_LIST_REORDER");
      Vem.selectedVehicleToMoveIndex = nil;
   end
   function Vem:renderVehiclePositionChangeInfo(vehicle)
      if Actions.states.reoderVehicleListMode then
         local editHeight = vehicle.widthPercent * 50;
         local editWidth = vehicle.width - (vehicle.widthPercent * 8);
         local editPosX = vehicle.posX + vehicle.widthPercent * 4;
         local editPosY = vehicle.posY + (vehicle.height - editHeight) - (vehicle.widthPercent * 12);
         local editOverlay = Overlay.new(g_baseUIFilename, editPosX, editPosY, editWidth, editHeight);
         local bgColor = {0.8, 0.8, 0.8, 0.6};
         local textSize = getCorrectTextSize(0.0112) * ConfigGui.modConfig.hudSizeMultiplier;
         local label = tostring(vehicle.counter);
         if Vem.selectedVehicleToMoveIndex == vehicle.index then
            bgColor = QS.Const.defaultGameColor;
            label = I18n:getLabel("VEHICLES_LIST_SELECTED");
         end
         editOverlay:setUVs(g_colorBgUVs)
         editOverlay:setColor(table.unpack(bgColor));
         editOverlay:render();
         Hud:renderBoxLabel({posX=editPosX, posY=editPosY, width=editWidth, height=editHeight, textSize=textSize, textColor=QS.Const.fullBlackColor, label=label});
      end
   end
   function Vem:onVehiclePositionChangeClick(vehicle)
      if Vem.selectedVehicleToMoveIndex then
         Vem:reorderEnterableVehicle(Vem.selectedVehicleToMoveIndex, vehicle.index, vehicle.instance); 
         Hud:updateSelectableVehiclesList(true);
         Vem.selectedVehicleToMoveIndex = nil;
      else
         Vem.selectedVehicleToMoveIndex = vehicle.index;
      end
   end
   function Vem:reorderEnterableVehicle(fromIndex, toIndex, vehicleToReplace)
      local vehicles = g_currentMission.vehicleSystem.vehicles;
      local enterables = g_currentMission.vehicleSystem.enterables;
      local vehicleToMove = enterables[fromIndex];
      local currentEnterableCount = 0;
      local vehicleIndexToRem = nil;
      local vehicleIndexToAdd = nil;
      table.remove(enterables, fromIndex);               
      table.insert(enterables, toIndex, vehicleToMove);  
      for index, vehicle in ipairs(vehicles) do
         if Vem:isAllowedVehicleType(vehicle) then
            currentEnterableCount = currentEnterableCount + 1;
            if vehicle.uniqueId == vehicleToReplace.uniqueId then
               vehicleIndexToAdd = index; 
            end
            if vehicle.uniqueId == vehicleToMove.uniqueId then
               vehicleIndexToRem = index; 
            end
         end
      end
      if vehicleIndexToRem and vehicleIndexToAdd then
         table.remove(vehicles, vehicleIndexToRem);                  
         table.insert(vehicles, vehicleIndexToAdd, vehicleToMove);   
      end
   end
   function Vem:onEnterVehicle(vehicle)
      QS:registerActionEventsHudToggle();
      function Vem.onEnterVehicleTimer()
         g_inputBinding:beginActionEventsModification(Vehicle.INPUT_CONTEXT_NAME); 
         QS:registerActionEventsHudToggle();
         g_inputBinding:endActionEventsModification();
         if Vem:isAllowedVehicleType(vehicle) then
            QS.currentVehicleContext = vehicle; 
         end
         if Hud.isVisible and Vem:isAllowedVehicleType(vehicle) then
            QS:registerActionEvents();
            Actions.updateStates();
            Hud:updateVehicleInfoList(vehicle);
            Hud:parseSelectedVehicleImplements();
         end
         QS:lockInputContext(true); 	      
      end
      addTimer(99, "onEnterVehicleTimer", Vem);
   end
   function Vem:onLeaveVehicle(vehicle)
      function Vem.onLeaveVehicleTimer()
         if Hud.isVisible and not g_localPlayer:getCurrentVehicle() then 
            Actions.updateStates(); 		         
            Hud:updateVehicleInfoList();           
            Hud:parseSelectedVehicleImplements();  
         end
         QS:lockInputContext(true);	 	      
      end
      addTimer(50, "onLeaveVehicleTimer", Vem);
   end
   function Vem:onVehicleAdded()
      Hud.shouldUpdateVehiclesList = true;
   end
   function Vem:onVehicleRemoved()
      Hud.shouldUpdateVehiclesList = true;
   end
   Player.onEnterVehicle = Utils.appendedFunction(Player.onEnterVehicle, Vem.onEnterVehicle);
   Player.onLeaveVehicle = Utils.appendedFunction(Player.onLeaveVehicle, Vem.onLeaveVehicle);
   g_messageCenter:subscribe(MessageType.VEHICLE_ADDED, Vem.onVehicleAdded, Vem);
   g_messageCenter:subscribe(MessageType.VEHICLE_REMOVED, Vem.onVehicleRemoved, Vem);
end
QuickSelector.init(QuickSelector.getDefaultNamespaces());