--|
--| [ Quick Selector ]
--|
--| @author			Quity - T. Florin
--| @Description:	Quick Selector Module - Game and Vehicle Actions
--|
--| Copyright (C) - T. Florin, All Rights Reserved.
--|
function QuickSelector.Actions.init(QS, Hud, Vem, Actions, ConfigGui, I18n, Utility, Dialogs)
	local vehicle;
	local selectedImplement;
	local Const = {
		lights = {
			ALL_OFF = 0,
			NO_LIGHT_MASKS_LIST = {[0]=true, [256]=true, [512]=true, [1024]=true, [2048]=true}
		}
	}
	function _getCurrentVehicle()
		if g_localPlayer ~= nil and g_localPlayer:getCurrentVehicle() ~= nil and Vem:isAllowedVehicleType(g_localPlayer:getCurrentVehicle()) then
			vehicle = g_localPlayer:getCurrentVehicle();
		else
			vehicle = nil;
		end
		return vehicle;
	end
	function _getSelectedImplement()
		local attachedImplements = vehicle and vehicle:getAttachedImplements() or nil;
		selectedImplement = nil;
		function _recursiveParseImplements(list)
			for _, implementRoot in pairs(list) do
				if not selectedImplement and implementRoot.object:getIsSelected() and not implementRoot.object.spec_attachable.detachingInProgress then
					selectedImplement = implementRoot.object;
					return;
				elseif not selectedImplement and implementRoot.object:getAttachedImplements() then
					_recursiveParseImplements(implementRoot.object:getAttachedImplements());
				end
			end
		end
		if vehicle and vehicle:getSelectedImplement() then
			selectedImplement = vehicle:getSelectedImplement().object;
		elseif attachedImplements then
			_recursiveParseImplements(attachedImplements);
		end
		return selectedImplement;
	end
	function _isTipAnimationInProgress()
		if selectedImplement and selectedImplement.spec_trailer then
			local tipState = selectedImplement:getTipState()
			return tipState ~= Trailer.TIPSTATE_CLOSED and tipState ~= Trailer.TIPSTATE_CLOSING;
		elseif vehicle and vehicle.spec_pipe then
			local unloadingState, _ = next(vehicle.spec_pipe.unloadingStates)
			return not vehicle:getIsPipeStateChangeAllowed(unloadingState);
		else
			return false;
		end
	end
	function _isTipOnGround(implementOrVehicle)
		if implementOrVehicle.spec_dischargeable then
			return implementOrVehicle.spec_dischargeable.currentDischargeState == Dischargeable.DISCHARGE_STATE_GROUND;
		end
		return false;
	end
	function _startMotor()
		if not Actions.states.startMotor and vehicle then
			vehicle:startMotor();
			Actions.states.startMotor = true;
			Hud.quickButtonsListNamesMap.START_ENGINE:applyButtonState(); 
		end
	end
	local SaveGM = {};
	function SaveGM:onSaveComplete(superFunc, dt)
		self.doSaveGameState = InGameMenu.SAVE_STATE_NONE;
		self.isSaving = false;
		g_gui:closeDialogByName("MessageDialog");
		return superFunc(self, dt);
	end
	InGameMenu.notifySaveComplete = Utils.overwrittenFunction(InGameMenu.notifySaveComplete, SaveGM.onSaveComplete);
	function Actions.updateStates()
		local currentLightMask;
		if _getCurrentVehicle() then
			_getSelectedImplement()
			currentLightMask = vehicle and vehicle.spec_lights and vehicle.spec_lights.lightsTypesMask or false;
		end
		local implementOrVehicle = selectedImplement or vehicle;
		function _isLightOn(lightMask)
			return currentLightMask and currentLightMask == bitOR(currentLightMask, 2 ^ lightMask) or false;
		end
		Actions.states = {
			startMotor = vehicle and vehicle:getIsMotorStarted() or false,
			activateHelper = vehicle and vehicle:getIsAIActive() or false,
			activateCruiseControl = vehicle and vehicle.spec_drivable.cruiseControl.state ~= Drivable.CRUISECONTROL_STATE_OFF or false,
			cruiseControlMaxSpeed = vehicle and vehicle.spec_drivable.cruiseControl.speed == vehicle.spec_drivable.cruiseControl.maxSpeed,
			enterExitVehicle = vehicle and true or false,
			lightsToggle = currentLightMask and Const.lights.NO_LIGHT_MASKS_LIST[currentLightMask] ~= true or false,
			lightsAll = currentLightMask and currentLightMask >= vehicle.spec_lights.maxLightStateMask or false,
			lightsFront = _isLightOn(Lights.LIGHT_TYPE_DEFAULT),
			lightsHighBeam = _isLightOn(Lights.LIGHT_TYPE_HIGHBEAM),
			lightsFrontTop = _isLightOn(Lights.LIGHT_TYPE_WORK_FRONT),
			lightsBack = _isLightOn(Lights.LIGHT_TYPE_WORK_BACK),
			lightsBeacon = vehicle and vehicle.spec_lights and vehicle.spec_lights.beaconLightsActive or false,
			lightsHazard = vehicle and vehicle.spec_lights.turnLightState == Lights.TURNLIGHT_HAZARD or false,
			implementAttachToggle = selectedImplement and true or false;
			implementPowerOn = implementOrVehicle and implementOrVehicle.getIsTurnedOn and implementOrVehicle:getIsTurnedOn() or false,
			implementFolding = implementOrVehicle and implementOrVehicle.spec_foldable and #implementOrVehicle.spec_foldable.foldingParts > 0 and implementOrVehicle:getIsUnfolded() or false,
			implementLowering = implementOrVehicle and implementOrVehicle:getIsLowered() or false,
			implementUnload = implementOrVehicle and _isTipAnimationInProgress() and not _isTipOnGround(implementOrVehicle);
			implementUnloadHere = implementOrVehicle and _isTipAnimationInProgress() and _isTipOnGround(implementOrVehicle);
		}
		Hud:updateSelectorsIcon();
	end
	function Actions.prerequisiteEnterable(btn, functionName)
		if _getCurrentVehicle() or (g_currentMission.interactiveVehicleInRange and g_currentMission.interactiveVehicleInRange:getIsEnterable()) then
			Actions.vehicles[functionName](functionName);
		end
	end
	function Actions.prerequisiteVehicle(btn, functionName)
		if _getCurrentVehicle() then
			Actions.vehicles[functionName](functionName);
		end
	end
	function Actions.prerequisiteImplement(btn, functionName)
		if _getCurrentVehicle() and _getSelectedImplement() then
			Actions.vehicles[functionName](functionName);
		end
	end
	Actions.general = {
		reoderVehicleListMode = function(btn, functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			Vem:toggleReoderVehiclesList(btn);
		end,
		openShop = function(btn)
			QS:hideHud();
			g_gui:changeScreen(nil, ShopMenu);
		end,
		openMenu = function(btn)
			QS:hideHud();
			g_gui:changeScreen(nil, InGameMenu); 
		end,
		saveGame = function(btn)
			g_inGameMenu:onButtonSaveGame();
		end,
		quitGame = function(btn)
			Dialogs.general:new("QuitDialog", "QUICKSELECTOR_GENERAL_QUIT_GAME_TITLE", {
				body = { x=0.5, y=0.5, w=460, h=160 },
				hasBackdrop = true,
				internalContext = QS.Const.InternalContexts.INFO_DIALOG,
				contentText = I18n:getName("QUICKSELECTOR_GENERAL_MESSAGE_EXIT_WARNING"),
				buttons = {
					{ name=I18n:getLabel("GENERAL_QUIT"), centerPosition=true, callback=function(self) executeConsoleCommand("qws"); self:close(); end },
					{ name=I18n:getLabel("GENERAL_EXIT"), centerPosition=true, callback=function(self) requestExit(); self:close(); end },
					{ name=I18n:getLabel("GENERAL_CANCEL"), callback=function(self) self:close();  end },
				}
			});
		end,
		showConfig = function(btn)
			ConfigGui:showDialog();
		end,
		closeHud = function(btn)
			QS:hideHud();
		end,
		restartGame = function()
			QS.commandRestartSaveGame();
		end
	};
	Actions.vehicles = {
		startMotor = function(functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			if Actions.states[functionName] then
				vehicle:startMotor();
			else
				vehicle:stopMotor();
			end
		end,
		enterExitVehicle = function(functionName)
			if Actions.states[functionName] and _getCurrentVehicle() then
				Actions.states[functionName] = false;
				g_localPlayer:leaveVehicle()
			elseif not Actions.states[functionName] and g_currentMission.interactiveVehicleInRange and g_currentMission.interactiveVehicleInRange:getIsEnterable() then
				Actions.states[functionName] = true;
				QS:revertLockedContext();
				g_localPlayer:requestToEnterVehicle(g_currentMission.interactiveVehicleInRange);
			end
		end,
		openAIScreen = function(_)
			local currentVehicle = _getCurrentVehicle();
			currentVehicle.spec_aiModeSelection.modeChangeStartTime = -99999;
			AIModeSelection.actionEventToggleAIState(currentVehicle, nil, 1); 
		end,
		activateHelper = function(functionName)
			Actions.states[functionName] = not vehicle:getIsAIActive();
			vehicle:toggleAIVehicle();
		end,
		activateCruiseControl = function(functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			vehicle:setCruiseControlState(Actions.states[functionName] and Drivable.CRUISECONTROL_STATE_ACTIVE or Drivable.CRUISECONTROL_STATE_OFF);
		end,
		increaseCruiseControl = function()
			vehicle:setCruiseControlMaxSpeed(vehicle.spec_drivable.cruiseControl.speed + 1);
		end,
		decreaseCruiseControl = function()
			vehicle:setCruiseControlMaxSpeed(vehicle.spec_drivable.cruiseControl.speed - 1);
		end,
		cruiseControlMaxSpeed = function(functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			vehicle:setCruiseControlMaxSpeed(vehicle.spec_drivable.cruiseControl.maxSpeed);
		end,
		lightsToggle = function()
			Lights.actionEventToggleLights(vehicle);
			Actions.updateStates();
		end,
		lightsAll = function(functionName)
			vehicle:setLightsTypesMask(not Actions.states[functionName] and vehicle.spec_lights.maxLightStateMask or Const.lights.ALL_OFF);
			Actions.updateStates();
		end,
		lightsFront = function()
			Lights.actionEventToggleLightFront(vehicle);
			Actions.updateStates();
		end,
		lightsHighBeam = function()
			if not Actions.states.lightsFront then
				Lights.actionEventToggleLightFront(vehicle);
			end
			Lights.actionEventToggleHighBeamLight(vehicle);
			Actions.updateStates();
		end,
		lightsFrontTop = function()
			Lights.actionEventToggleWorkLightFront(vehicle);
			Actions.updateStates();
		end,
		lightsBack = function()
			Lights.actionEventToggleWorkLightBack(vehicle);
			Actions.updateStates();
		end,
		lightsBeacon = function(functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			vehicle:setBeaconLightsVisibility(Actions.states[functionName]);
		end,
		lightsHazard = function(functionName)
			Actions.states[functionName] = not Actions.states[functionName];
			vehicle:setTurnLightState(Actions.states[functionName] and Lights.TURNLIGHT_HAZARD or Lights.TURNLIGHT_OFF);
		end,
		implementAttachToggle = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			Actions.states[functionName] = not Actions.states[functionName];
			if implementOrVehicle.spec_attacherJoints and next(implementOrVehicle.spec_attacherJoints.attachableInfo) then
				QS:temporaryRevertLockedContext("onImplementAttachTimer");
			elseif selectedImplement then
				Actions.attachTogglePostDetach = function()
					QS:temporaryRevertLockedContext("onImplementDetachTimer");
				end
				if not selectedImplement.hasCustomPostDetachAppendedFn then
					selectedImplement.hasCustomPostDetachAppendedFn = true;
					selectedImplement.postDetach = Utils.appendedFunction(selectedImplement.postDetach, Actions.attachTogglePostDetach);
				end
			end
			AttacherJoints.detachAttachedImplement(vehicle);
			Hud:parseSelectedVehicleImplements();
		end,
		implementPowerOn = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			_startMotor();
			Actions.states[functionName] = not Actions.states[functionName];
			if Actions.states[functionName] and implementOrVehicle.getCanBeTurnedOn and implementOrVehicle:getCanBeTurnedOn() then
				implementOrVehicle:setIsTurnedOn(true);
			elseif implementOrVehicle.getCanBeTurnedOn then
				implementOrVehicle:setIsTurnedOn(false);
			end
		end,
		implementFolding = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			if implementOrVehicle.spec_foldable then
				_startMotor();
				Actions.states[functionName] = not Actions.states[functionName];
				Foldable.actionEventFold(implementOrVehicle);
			end
		end,
		implementLowering = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			local isFoldable = implementOrVehicle.spec_foldable and #implementOrVehicle.spec_foldable.foldingParts > 0 or false;
			local isPickup = implementOrVehicle.spec_pickup;
			Actions.states[functionName] = not Actions.states[functionName];
			_startMotor();
			if isFoldable and implementOrVehicle:getIsFoldMiddleAllowed() then
				Foldable.actionEventFoldMiddle(implementOrVehicle);
			end
			if isPickup then
				implementOrVehicle:setPickupState(not implementOrVehicle.spec_pickup.isLowered)
			else
				vehicle.rootVehicle:handleLowerImplementEvent();
			end
		end,
		implementUnload = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			Actions.states[functionName] = not _isTipAnimationInProgress();
			if vehicle.spec_pipe and vehicle.spec_pipe.hasMovablePipe then
				Pipe.actionEventTogglePipe(vehicle);
			elseif selectedImplement and selectedImplement.spec_pipe and selectedImplement.spec_pipe.hasMovablePipe then
				Pipe.actionEventTogglePipe(selectedImplement);
			end
			if implementOrVehicle.spec_baler and not implementOrVehicle.spec_baler.hasPlatform then
				implementOrVehicle.spec_baler:handleUnloadingBaleEvent();
			elseif implementOrVehicle.spec_baler then
				implementOrVehicle.spec_baler:dropBaleFromPlatform(false);
			end
			if implementOrVehicle.spec_baleWrapper then
				BaleWrapper.actionEventDrop(implementOrVehicle);
			end
			if implementOrVehicle.getCanToggleDischargeToObject and implementOrVehicle:getCanToggleDischargeToObject() then
				Dischargeable.actionEventToggleDischarging(implementOrVehicle);
			end
		end,
		implementUnloadHere = function(functionName)
			local implementOrVehicle = selectedImplement or vehicle;
			Actions.states[functionName] = not _isTipAnimationInProgress();
			if vehicle.spec_pipe and vehicle.spec_pipe.hasMovablePipe then
				Pipe.actionEventToggleDischargeToGround(vehicle);
			elseif selectedImplement and selectedImplement.spec_pipe and selectedImplement.spec_pipe.hasMovablePipe then
				Pipe.actionEventToggleDischargeToGround(selectedImplement);
			end
			if implementOrVehicle.getCanToggleDischargeToGround and implementOrVehicle:getCanToggleDischargeToGround() then
				Dischargeable.actionEventToggleDischargeToGround(implementOrVehicle);
			end
		end
	}
	Actions.updateStates();
end