--|
--| [ Quick Selector ]
--|
--| @author			Quity - T. Florin
--| @Description:	Quick Selector Module - Configuration
--|
--| Copyright (C) - T. Florin, All Rights Reserved.
--|
function QuickSelector.ConfigGui.init(QS, Hud, Vem, Actions, ConfigGui, I18n, Utility, Dialogs)
	local xmlConfigFile = getUserProfileAppPath() .. "modSettings/quickSelectorConfig.xml";
	local changedConfig = {};
	ConfigGui.dialog = {};
	ConfigGui.isVisible = false;
	ConfigGui.defaultConfig = {
		hudSizeMultiplier = 0.7,							
		vehicleListMaxPerPage = 16,						
		vehicleDisplayInfoLine = true,					
		doubleRightMouseClickDisplayHud = true,		
		rightMouseHoldDisplayHud = false,				
		enableGamepadControl = true,						
		gamepadUseLeftAnalogToSteer = false,			
		gamepadDeadzoneLimit = 0.05,						
		gamepadAnalogsSensitivity = 0.2,					
	};
	function ConfigGui.loadConfiguration()
		local configXML = loadXMLFile("QuickSelectorConfigXML", xmlConfigFile);
		for id, _ in pairs(ConfigGui.defaultConfig) do
			local valueString = getXMLString(configXML, "configuration." .. id);
			local loadedValue = ConfigGui.defaultConfig[id];
			if valueString == "true" or valueString == "false" then
				loadedValue = valueString == "true" and true or false;
			elseif tonumber(valueString) ~= nil then
				loadedValue = tonumber(valueString);
			end
			ConfigGui.modConfig[id] = loadedValue;
		end
		ConfigGui.modConfig.vehicleListMaxPerPage = _getVehicleListMaxPerPageRange(ConfigGui.modConfig.hudSizeMultiplier); 
		delete(configXML);
	end
	function ConfigGui.saveConfiguration(config, configXML)
		if not configXML then
			configXML = loadXMLFile("QuickSelectorConfigXML", xmlConfigFile);
		end
		for id, _ in pairs(ConfigGui.defaultConfig) do
			if config[id] == nil then
				config[id] = ConfigGui.defaultConfig[id]; 
			end
			ConfigGui.modConfig[id] = config[id];
			setXMLString(configXML, "configuration." .. id, tostring(ConfigGui.modConfig[id]));
		end
		changedConfig = {};
		saveXMLFile(configXML);
		delete(configXML);
	end
	function _getVehicleListMaxPerPageRange(hudSizeMultiplierValue)
		local interItemSpacingPx = getNormalizedScreenValues(-0.0004 * g_referenceScreenWidth * hudSizeMultiplierValue, 0); 
		local vehicleBoxNorm, _ = getNormalizedScreenValues(QS.Const.VEHICLE_BOX_PIXEL_SIZE * hudSizeMultiplierValue, 0);
		local sideElementsNorm, _ = getNormalizedScreenValues(700, 0);   
		local usableNorm = 1.0 - sideElementsNorm;
		local entryNorm = vehicleBoxNorm + interItemSpacingPx;
		return  math.floor(usableNorm / entryNorm + 0.5);
	end
	ConfigGui.modConfig = ConfigGui.defaultConfig;
	if fileExists(xmlConfigFile) then
		ConfigGui.loadConfiguration();
	else
		ConfigGui.saveConfiguration(ConfigGui.defaultConfig, createXMLFile("QuickSelectorConfigXML", xmlConfigFile, "configuration"));
	end
	function _createContentOptionsList(dg)
		local list = {};
		local startX = dg.body.posX;
		local startTopY = dg.contentEndY;
		local currentPosY = startTopY; 
		local elementWidth = dg.body.width;
		local _, elementHeight = getNormalizedScreenValues(0, 40);
		local elementWidthPercent = elementWidth / 100;
		local elementHeightPercent = elementHeight / 100;
		local elementBgColor = {0.016, 0.016, 0.016, 1};
		local elementBgColorEven = {0.024, 0.024, 0.024, 1};
		local bodyPadding = getNormalizedScreenValues(20, 0);
		local valuePosX = startX + elementWidthPercent * 33;
		local valueWidth = elementWidthPercent * 12;
		local descrWidth = elementWidthPercent * 49;
		local arrowWidth, arrowHeight = getNormalizedScreenValues(10, 10);
		local optionsList = {
			{id = "hudSizeMultiplier", min = 0.5, max = 1.1, steps = 0.1, multiplier = 100, normalizeRenderedValue = 0.3 },
			{id = "vehicleListMaxPerPage", min = 8, max = _getVehicleListMaxPerPageRange(ConfigGui.modConfig.hudSizeMultiplier), steps = 1, multiplier = 1, isScale = true }, 
			{id = "vehicleDisplayInfoLine"},
			{id = "doubleRightMouseClickDisplayHud"},
			{id = "rightMouseHoldDisplayHud"},
			{id = "enableGamepadControl"},
			{id = "gamepadUseLeftAnalogToSteer"},
			{id = "gamepadDeadzoneLimit", min = 0.01, max = 0.2, steps = 0.01, multiplier = 100 },
			{id = "gamepadAnalogsSensitivity", min = 0.1, max = 1, steps = 0.1, multiplier = 100 },
		};
		function _getOptionsIndexById(targetId)
			for i, element in ipairs(optionsList) do
				if element.id == targetId then
					return i;
				end
			end
		end
		function _createArrowBtn(index, isLeftArrow, configItem, arrowPosX, arrowWidth, arrowHeight)
			local arrowBtn = Hud:createBtn(index, false, true, false, arrowPosX, currentPosY - (arrowHeight / 2), arrowWidth, arrowHeight, 0, 9, "configArrow_"..tostring(isLeftArrow), {
				internalContext = QS.Const.InternalContexts.CONFIG_GUI,
				bgColor = QS.Const.fullWhiteColorShade0_5,
				defaultBgColor = QS.Const.fullWhiteColorShade0_5,
				hoveredBgColor = QS.Const.defaultGameColorShade0_2,
				disabledBgColor = QS.Const.defaultGameColorShade0_1,
				onClickCallback = function(btn)
					btn.bgColor = QS.Const.activeClickColor;
					configItem:onValueChange(isLeftArrow);
				end,
				onLeftClickCallback = function(self)
					self.bgColor = self.isHovered and self.hoveredBgColor or self.defaultBgColor;
				end,
				onHoverCallback = function(self)
					self.bgColor = self.hoveredBgColor;
				end,
				onLeftHoverCallback = function(self)
					self.bgColor = self.defaultBgColor;
				end,
				renderBtn = function(self)
					self.overlayBg:setColor(table.unpack(self.isDisabled and self.disabledBgColor or self.bgColor));
					self.overlayBg:render();
				end,
			});
			arrowBtn.overlayBg = Utility.newColoredOverlay(QS.assetsDir .. "arrowLeftFull.dds", arrowPosX, currentPosY - (arrowHeight / 2), arrowWidth, arrowHeight, arrowBtn.defaultBgColor),
			table.insert(dg.activeButtonsList, arrowBtn);
			return arrowBtn;
		end
		function _updateArrowsState(optionItem, option, value)
			if option.min and option.max then
				optionItem.arrowLeft.isDisabled = value <= option.min;
				optionItem.arrowRight.isDisabled = value >= option.max;
			end
		end
		function _updateVehicleListMaxPerPageRange()
			local newVehicleMax = _getVehicleListMaxPerPageRange(changedConfig.hudSizeMultiplier);
			local currentValue = changedConfig.vehicleListMaxPerPage and changedConfig.vehicleListMaxPerPage or ConfigGui.modConfig.vehicleListMaxPerPage;
			local optionConfig = optionsList[_getOptionsIndexById("vehicleListMaxPerPage")];
			changedConfig.vehicleListMaxPerPage = currentValue > newVehicleMax and newVehicleMax or currentValue;
			optionConfig.max = newVehicleMax;
			_updateArrowsState(list.vehicleListMaxPerPage, optionConfig, currentValue);
		end
		function _updateDependentOptions(changedOptionItem, newValue)
			if changedOptionItem.id == "hudSizeMultiplier" then
				_updateVehicleListMaxPerPageRange();
			end
			if changedOptionItem.id == "rightMouseHoldDisplayHud" and newValue == true then
				changedConfig.doubleRightMouseClickDisplayHud = false;
			end
			if changedOptionItem.id == "doubleRightMouseClickDisplayHud" and newValue == true then
				changedConfig.rightMouseHoldDisplayHud = false;
			end
		end
		for index, option in ipairs(optionsList) do
			local value = ConfigGui.modConfig[option.id];
			local arrowBtnSpacing = getNormalizedScreenValues(6, 0);
			currentPosY = currentPosY - elementHeight;
			local optionItem = {
				id = option.id,
				index = index,
				posX = startX,
				posY = currentPosY,
				width = elementWidth,
				height = elementHeight,
				label = I18n:getName("QS_SETTINGS_" .. Utility.toConstantCase(option.id) .. "_LABEL"),
				labelSize = getCorrectTextSize(0.013),
				description = I18n:getName("QS_SETTINGS_" .. Utility.toConstantCase(option.id) .. "_DESCR"),
				descrPosX = valuePosX + valueWidth + elementWidthPercent * 5,
				descrSize = getCorrectTextSize(0.011),
				value = value,
				valueBox = {
					posX = valuePosX,
					posY = currentPosY - (elementHeightPercent * 35),
					width = valueWidth,
					height = elementHeight - (elementHeightPercent * 32),
					bgColor = {0.04, 0.04, 0.04, 0.96},
				},
				renderValue = function(self)
					local extraLabelPosY = 0.001;
					local updatedValue = changedConfig[self.id];
					local valueStr;
					if updatedValue == nil then
						updatedValue = self.value;
					end
					if option.normalizeRenderedValue then
						updatedValue = updatedValue + option.normalizeRenderedValue;
					end
					if type(updatedValue) == "boolean" then
						valueStr = (updatedValue and g_i18n:getText("ui_on") or g_i18n:getText("ui_off")):upper();
					elseif not option.isScale then
						valueStr = MathUtil.round(updatedValue * option.multiplier) .. "%";
					else
						valueStr = updatedValue;
					end
					setTextColor(table.unpack(QS.Const.fullWhiteColor));
					setTextAlignment(RenderText.ALIGN_CENTER);
					renderText(self.valueBox.posX + self.valueBox.width / 2, self.posY + extraLabelPosY, self.labelSize, tostring(valueStr)); 
				end,
				renderDescr = function(self)
					setTextColor(table.unpack(QS.Const.fullWhiteColorShade0_5));
					setTextAlignment(RenderText.ALIGN_LEFT);
					setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_MIDDLE);
					setTextWrapWidth(descrWidth);
					setTextLineBounds(0, 2);
					renderText(self.descrPosX, self.posY, self.descrSize, self.description); 
				end,
				onValueChange = function(self, lowerValue)
					local updatedValue = changedConfig[self.id];
					local newValue;
					if updatedValue == nil then
						updatedValue = self.value;
					end
					if type(updatedValue) == "boolean" then
						newValue = not updatedValue;
					else
						newValue = MathUtil.round(lowerValue and (updatedValue - option.steps) or (updatedValue + option.steps), 2);
					end
					changedConfig[self.id] = newValue;
					_updateDependentOptions(self, newValue);
					_updateArrowsState(self, option, newValue);
				end,
			};
			local boxBorderW, boxBorderH = getNormalizedScreenValues(3, 3);
			optionItem.valContBox = Hud:createBtn(index, false, true, false, optionItem.valueBox.posX-boxBorderW/2, optionItem.valueBox.posY-boxBorderH/2, optionItem.valueBox.width+boxBorderW, optionItem.valueBox.height+boxBorderH, 0, 8, "optionValBg", {
				internalContext = QS.Const.InternalContexts.CONFIG_GUI,
				bgColor = QS.Const.fullWhiteColorShade0_1,
				onHoverCallback = function(self)
					self.bgColor = QS.Const.defaultGameColor;
				end,
				onLeftHoverCallback = function(self)
					if not optionItem.arrowLeft.isHovered and not optionItem.arrowRight.isHovered then
						self.bgColor = QS.Const.fullWhiteColorShade0_1;
					end
				end,
				renderBox = function(self, el)
					local elBgColor = el.index % 2 == 0 and elementBgColorEven or elementBgColor;
					drawFilledRect(el.posX, el.posY - (el.height / 2), el.width, el.height, table.unpack(elBgColor)); 
					setTextAlignment(RenderText.ALIGN_LEFT);
					setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_MIDDLE);
					setTextColor(table.unpack(QS.Const.fullWhiteColor));
					renderText(el.posX + bodyPadding, el.posY, el.labelSize, el.label); 
					drawFilledRectRound(self.posX, self.posY, self.width, self.height, 0.2, table.unpack(self.bgColor)); 
					drawFilledRectRound(el.valueBox.posX, el.valueBox.posY, el.valueBox.width, el.valueBox.height, 0.2, table.unpack(el.valueBox.bgColor)); 
				end,
			});
			table.insert(dg.activeButtonsList, optionItem.valContBox);
			optionItem.arrowLeft = _createArrowBtn(index, true, optionItem, valuePosX + arrowBtnSpacing, arrowWidth, arrowHeight);
			optionItem.arrowRight = _createArrowBtn(index, false, optionItem, valuePosX + valueWidth - arrowWidth - arrowBtnSpacing, arrowWidth, arrowHeight);
			optionItem.arrowRight.overlayBg:setRotation(math.rad(180), arrowWidth / 2, arrowHeight / 2);
			_updateArrowsState(optionItem, option, optionItem.value);
			list[option.id] = optionItem;
		end
		return list;
	end
	function ConfigGui.createGui()
		local dg;
		function _save()
			ConfigGui.saveConfiguration(changedConfig);
			Hud:createQuickButtons();							
			Hud:updateSelectableVehiclesList(false);		
			Hud:updateVehicleInfoList();						
			Hud:parseSelectedVehicleImplements();			
			Hud:updateSelectorsIcon();
			ConfigGui.hideDialog();
		end
		dg = Dialogs.general:new("ConfigGui", "QS_SETTINGS_SCREEN_TITLE", {
			body = { x=0.5, y=0.5, w=1200, h=460 },
			hasBackdrop = true,
			internalContext = QS.Const.InternalContexts.CONFIG_GUI,
			buttons = {
				{ name=I18n:getLabel("GENERAL_SAVE"), callback=_save },
				{ name=I18n:getLabel("GENERAL_CANCEL"), callback=ConfigGui.hideDialog },
			},
			callbacks = {
				close = ConfigGui.hideDialog,
				renderContent = function(dg)
					for id, el in pairs(dg.contentOptionsList) do
						el.valContBox:renderBox(el);
						el.arrowLeft:renderBtn();
						el:renderValue(); 
						el.arrowRight:renderBtn();
						el:renderDescr(); 
					end
				end;
			}
		});
		dg.contentOptionsList = _createContentOptionsList(dg);
		ConfigGui.dialog = dg;
	end
	function ConfigGui:showDialog()
		ConfigGui.createGui();
		ConfigGui.isVisible = true;
	end
	function ConfigGui:hideDialog()
		ConfigGui.isVisible = false;
		ConfigGui.dialog:delete();
		ConfigGui.dialog = nil;
		changedConfig = {};
	end
end